// DITLVIEW GLUT (Rewrite of DITLView) - Jim Bumgardner
//
// To add: add dragging of dialog on screen.
//         add scaling

#include "MandelZoom.h"

#include <stdarg.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>
#include <time.h>

// Control this with an oversampling control...

// Note: 256 works nicely for a realtime zoom..
#define kDefWidth	1024


MandelWin	*gMandelWin;

MandelWin::MandelWin(int width, int height)
{
	gMandelWin = this;
	errMsgBuf[0] = 0;
	b_IsFullscreen = false;

	recurseFactor = 160;
	cutOff = 4;

	cx = -0.5;
	cy = 0.000001;
	rWidth = 4.0;
	jx = 0;
	jy = 0;
	startRender = 0;
	elapsedRender = 0;
	
	mx = my = 0;
	mouseDownFlags = 0;

	isZooming = false;


	srand(time(NULL) % 256);
	this->width = width;
	this->height = height;
#ifdef STANDALONE
	this->m_desiredFPS = 24;	// was 30
#else
	this->m_desiredFPS = 24;	// was 24
#endif
	this->m_sleepAdjust = 1000*2/(m_desiredFPS+3);

	glViewport(0,0,width,height);						// Reset The Current Viewport

	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glLoadIdentity();												// Reset The Projection Matrix

	// Calculate The Aspect Ratio Of The Window
	gluPerspective(90.0f,(GLfloat)width/(GLfloat)height,1.0f,20.0f);

	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix

	m_fFramesPerSec = 10.0f;

	InitTimer();

	
	if (true) {	// using depth buffer?
		glClearDepth(1.0f);									// Depth Buffer Setup
		glEnable(GL_DEPTH_TEST);							// Enables Depth Testing
		glDepthFunc(GL_LEQUAL);								// The Type Of Depth Testing To Do
		glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);	// Really Nice Perspective Calculations
	}
	fLastScopeStart= 0.0f;

	offWorld1 = new TextureImage(kDefWidth, kDefWidth, GL_RGBA);
	offWorld2 = new TextureImage(kDefWidth, kDefWidth, GL_RGBA);
	fgWorld = offWorld1;
	bgWorld = offWorld2;
	isJulia = 0;
	renderMode = RM_TigerCon; // RM_SineToR; // RM_Tiger; // RM_SineToR; // RM_BlackFill;
	paletteType = RM_HSVRamp; // RM_Grayscale; // RM_RGBRamp; // RM_HSVRamp; !!!
	cFreq[0] = 1;
	cFreq[1] = 2;
	cFreq[2] = 4;
	cPhase[0] = 0;
	cPhase[1] = 0;
	cPhase[2] = 0;
	cAdjustComponent = 0;

	InitColors();
	ResetRender();
	
	// UpdateTexture();
}

//  used by standalone app only
void MandelWin::initGL(int width, int height)
{
	resizeGL(width, height);
}

void MandelWin::resizeGL(int width, int height)
{
	this->width = width;
	this->height = height;
	glViewport(0,0,width,height);						// Reset The Current Viewport
	glMatrixMode(GL_PROJECTION);						// Select The Projection Matrix
	glLoadIdentity();												// Reset The Projection Matrix

	// Calculate The Aspect Ratio Of The Window
	gluPerspective(90.0f,(GLfloat)width/(GLfloat)height,1.0f,20.0f);
	glMatrixMode(GL_MODELVIEW);							// Select The Modelview Matrix
	ResetRender();
}



MandelWin::~MandelWin()
{
	delete offWorld1;
	delete offWorld2;

}

int MandelWin::display()
{
	glDrawBuffer(GL_BACK);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();

	gluLookAt( 0.0f, 0.0f,  5.5f , // eye pt
						 0.0f, 0.0f,  0.0f,  // target (center)
						 0.0f, 1.0f,  0.0f); // up vector

	UpdateFrame();

  glFlush();

	return true;	// true means swap the buffer
}

void MandelWin::DrawString(int x, int y, char *string)
{
   int len, i;

   glRasterPos2f(x, y);
   len = (int) strlen(string);
   for (i = 0; i < len; i++) {
     glutBitmapCharacter(GLUT_BITMAP_8_BY_13, string[i]);
	 }
 }


void MandelWin::DrawStats()
{
	char	tbuf[81];
	GLfloat	cx = 0, cy = 0, cw = 300, ch = 100;
	glDisable(GL_TEXTURE_2D);
	glBegin(GL_QUADS);
		glNormal3f(0, 0, 1);
				
		glColor4f(1,1,1,.25);
		glVertex3f(cx,  cy,   1.0f);	// Point 1 (Front)
		glVertex3f(cx+cw, cy,   1.0f);	// Point 2 (Front)
		glVertex3f(cx+cw, cy+ch,  1.0f);	// Point 3 (Front)
		glVertex3f(cx,  cy+ch,  1.0f);	// Point 4 (Front)
	glEnd();
	glEnable(GL_TEXTURE_2D);

	glColor4f(0,0,0,1);
#define LH	13
	int y = 0;
	sprintf(tbuf, "%5.2f fps", m_fFramesPerSec);	
	DrawString(20, ++y * LH, tbuf);
	sprintf(tbuf, "%.2f elapsed render", elapsedRender);
	DrawString(20, ++y * LH, tbuf);
	sprintf(tbuf, "%d resolution", resolution);
	DrawString(20, ++y * LH, tbuf);
	sprintf(tbuf, "%d, %d", mx, my);
	DrawString(20, ++y * LH, tbuf);
}

void MandelWin::UpdateFrame()
{
  UpdateTimer();

	glPushMatrix();
	// Perform drawing here...
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glDisable(GL_DEPTH_TEST);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );

	int		sw = glutGet(GLUT_WINDOW_WIDTH);
	int		sh = glutGet(GLUT_WINDOW_HEIGHT);
	
	GLfloat	cx = 0,cy = 0;
	GLfloat cw = sw;
	GLfloat ch = sh;
	GLfloat tx,ty,tw,th;
	GLfloat	alpha = 1;

	ViewOrtho();

	// glRotatef(m_fTime,0,0,1);
	GLfloat r,er;

	if (isZooming)
	{
		r = (m_fTime - startZoom) / zoomDuration;
		if (r > 1) {
			r = 1;
			isZooming = false;
		}
		er = r; // need to use function to provide linear zoom effect...
	}

	if (isZooming && !zoomOut) 
	{
		// this is a logarithmic zooming factor...
		GLfloat	zr = bgWorld->ecw / bgWorld->scw;
		er = (pow(zr,r) - 1) / (zr - 1);

		tx = ty = 0;
		tw = th = 1;

		cx = bgWorld->scx + bgWorld->dcx * er;
		cy = bgWorld->scy + bgWorld->dcy * er;
		cw = bgWorld->scw + bgWorld->dcw * er;
		ch = bgWorld->sch + bgWorld->dch * er;
		alpha = bgWorld->sa + bgWorld->da * er;

		glBindTexture(GL_TEXTURE_2D, bgWorld->texID);
		glBegin(GL_QUADS);
			glNormal3f(0, 0, 1);
			
			glColor4f(1,1,1,alpha);
			glTexCoord2f(tx,		ty);				glVertex3f(cx,  cy,   1.0f);	// Point 1 (Front)
			glTexCoord2f(tx+tw,  ty);				glVertex3f(cx+cw, cy,   1.0f);	// Point 2 (Front)
			glTexCoord2f(tx+tw,  ty+th);		glVertex3f(cx+cw, cy+ch,  1.0f);	// Point 3 (Front)
			glTexCoord2f(tx,		ty+th);			glVertex3f(cx,  cy+ch,  1.0f);	// Point 4 (Front)
		glEnd();
	}


	if (isZooming)
	{
		tx = ty = 0;
		tw = th = 1;
		GLfloat	zr = fgWorld->ecw / fgWorld->scw;
		er = (pow(zr,r) - 1) / (zr - 1);

		cx = fgWorld->scx + fgWorld->dcx * er;
		cy = fgWorld->scy + fgWorld->dcy * er;
		cw = fgWorld->scw + fgWorld->dcw * er;
		ch = fgWorld->sch + fgWorld->dch * er;
		alpha = fgWorld->sa + fgWorld->da * er; // was r*r
	}
	else 
	{
		r = er = 1;
		tx = ty = 0;
		tw = th = 1;
		cx = cy = 0;
		cw = sw;
		ch = sh;
		alpha = 1;
	}
	glBindTexture(GL_TEXTURE_2D, fgWorld->texID);
	glBegin(GL_QUADS);
		glNormal3f(0, 0, 1);
		
		glColor4f(1,1,1,alpha);
		glTexCoord2f(tx,		ty);			glVertex3f(cx,		cy,   1.0f);	// Point 1 (Front)
		glTexCoord2f(tx+tw,  ty);			glVertex3f(cx+cw, cy,   1.0f);	// Point 2 (Front)
		glTexCoord2f(tx+tw,  ty+th);	glVertex3f(cx+cw, cy+ch,  1.0f);	// Point 3 (Front)
		glTexCoord2f(tx,		ty+th);		glVertex3f(cx,		cy+ch,  1.0f);	// Point 4 (Front)

	glEnd();

	// When zooming out, show detailed area on top...
	if (isZooming && zoomOut)
	{
		tx = ty = 0;
		tw = th = 1;
		GLfloat	zr = bgWorld->ecw / bgWorld->scw;
		er = (pow(zr,r) - 1) / (zr - 1);

		cx = bgWorld->scx + bgWorld->dcx * er;
		cy = bgWorld->scy + bgWorld->dcy * er;
		// cw = bgWorld->scw * pow(bgWorld->rcw,r);
		// ch = bgWorld->sch * pow(bgWorld->rch,r);
		cw = bgWorld->scw + bgWorld->dcw * er;
		ch = bgWorld->sch + bgWorld->dch * er;
		alpha = bgWorld->sa + bgWorld->da * r*r*r; // was r*r

		glBindTexture(GL_TEXTURE_2D, bgWorld->texID);
		glBegin(GL_QUADS);
			glNormal3f(0, 0, 1);
			
			glColor4f(1,1,1,alpha);
			glTexCoord2f(tx,		ty);				glVertex3f(cx,  cy,   1.0f);	// Point 1 (Front)
			glTexCoord2f(tx+tw,  ty);				glVertex3f(cx+cw, cy,   1.0f);	// Point 2 (Front)
			glTexCoord2f(tx+tw,  ty+th);		glVertex3f(cx+cw, cy+ch,  1.0f);	// Point 3 (Front)
			glTexCoord2f(tx,		ty+th);			glVertex3f(cx,  cy+ch,  1.0f);	// Point 4 (Front)
		glEnd();
	}
	if (showStats)
		DrawStats();

	ViewPerspective();
	glPopMatrix();

}

#ifdef HAVE_GETTIMEODAY
static double startClock, lastClock;
#else
static long startClock, lastClock;
#endif
static bool gInitedStats;

// float gFrameRate, gTime;

void MandelWin::InitTimer()
{
    if (!gInitedStats)
    {
	gInitedStats = true;
#ifdef HAVE_GETTIMEODAY
        struct timeval ts;
        gettimeofday(&ts, NULL);
        startClock = ts.tv_sec + ts.tv_usec/1000000.0;
        lastClock = startClock;
#else
	startClock = timeGetTime();
	lastClock = timeGetTime();
#endif
	m_iFrames = 0;
	m_fFramesPerSec = 50;
	m_fTime = 0;
	m_fLastKey = 0;
    }
}

void MandelWin::UpdateTimer()
{
#ifdef HAVE_GETTIMEODAY
    double  rightNow;
    struct timeval ts;
    gettimeofday(&ts, NULL);
    rightNow = ts.tv_sec + ts.tv_usec/1000000.0;
    m_fTime = (rightNow - startClock);
    float dt = (rightNow - lastClock);
#else  
    long  rightNow;
    rightNow = timeGetTime();
    m_fTime = (rightNow - startClock)/1000.0f;
    float dt = (rightNow - lastClock)/1000.0f;
#endif    

    if (dt >= 1.0f) 
    {
			int lastFPS = (int) m_fFramesPerSec;
      m_fFramesPerSec = m_iFrames / dt;
      m_iFrames = 0;
      lastClock = rightNow;
      //printf("Framerate = %5.2f\n", gFrameRate);
			if (lastFPS > 0)
				m_sleepAdjust += (1000/m_desiredFPS - 1000/lastFPS) / 2;		
			if (m_sleepAdjust < 1)
				m_sleepAdjust = 1;
			else if (m_sleepAdjust > 1000)
				m_sleepAdjust = 1000;
		}
		else
			m_iFrames++;
}

bool MandelWin::HandleSpecialKey(int x)
{
	switch (x) {
	case GLUT_KEY_UP:			ScrollMandelbrot(0,-.25);		break;
	case GLUT_KEY_DOWN:		ScrollMandelbrot(0,.25);		break;
	case GLUT_KEY_LEFT:		ScrollMandelbrot(-.25,0);		break;
	case GLUT_KEY_RIGHT:	ScrollMandelbrot(.25,0);		break;
	case GLUT_KEY_PAGE_UP:	
			renderMode = (renderMode+1) % RM_NbrRenderModes;	
			ResetRender();
			break;
	case GLUT_KEY_PAGE_DOWN:
			paletteType = (paletteType+1) % kNbrPalettes;
			InitColors();
			break;
	case GLUT_KEY_HOME:
			recurseFactor = 160;
			cutOff = 4;

			cx = -0.5;
			cy = 0.000001;
			rWidth = 4.0;
			ResetRender();
			break;
	}
	return true;
}

void MandelWin::ToggleFullscreen()
{
	static int sWidth, sHeight, sPosX, sPosY;

	b_IsFullscreen = ! b_IsFullscreen;
	
	if (b_IsFullscreen)
	{
		sPosX = glutGet(GLUT_WINDOW_X);
		sPosY = glutGet(GLUT_WINDOW_Y);
		sWidth = glutGet(GLUT_WINDOW_WIDTH);
		sHeight = glutGet(GLUT_WINDOW_HEIGHT);
		glutFullScreen();
	}
	else {
		glutReshapeWindow(sWidth, sHeight);
		glutPositionWindow(sPosX, sPosY);
	}
	ResetRender();
}

void MandelWin::HandleMouseMove(int x, int y)
{
	mx = x;
	my = y;
}

void MandelWin::HandleMouseClick(int button, int state, int x, int y)
{
	mx = x;
	my = y;
	if (state == GLUT_UP)
	{
		switch (button)
		{
		case GLUT_LEFT_BUTTON:		mouseDownFlags &= ~kLeftButton;	break;
		case GLUT_MIDDLE_BUTTON:	mouseDownFlags &= ~kMiddleButton;	break;
		case GLUT_RIGHT_BUTTON:		mouseDownFlags &= ~kRightButton;	break;
		}
	}
	else {
		switch (button)
		{
		case GLUT_LEFT_BUTTON:		mouseDownFlags |= kLeftButton;	break;
		case GLUT_MIDDLE_BUTTON:	mouseDownFlags |= kMiddleButton;	break;
		case GLUT_RIGHT_BUTTON:		mouseDownFlags |= kRightButton;	break;
		}
	}
}


void MandelWin::ScrollMandelbrot(double x, double y)
{
	cx += x*rWidth;
	cy += y*hWidth;
	ResetRender();
}


#define kZoomFactor	2.0 // currently only works for 4...

void MandelWin::ZoominMandelbrot(int x, int y)
{
	int sWidth = glutGet(GLUT_WINDOW_WIDTH);
	int sHeight = glutGet(GLUT_WINDOW_HEIGHT);

	if (x < sWidth/(kZoomFactor*2))
			x = sWidth/(kZoomFactor*2);
	if (y < sHeight/(kZoomFactor*2))
			y = sHeight/(kZoomFactor*2);
	if (x > sWidth-sWidth/(kZoomFactor*2))
			x = sWidth-sWidth/(kZoomFactor*2);
	if (y > sHeight-sHeight/(kZoomFactor*2))
			y = sHeight-sHeight/(kZoomFactor*2);


//	if (x > sWidth*3/8 && x < sWidth*5/8)	x = sWidth/2;
//	if (y > sHeight*3/8 && y < sHeight*5/8) y = sHeight/2;

	double pxR = (double) x / sWidth;
	double pyR = (double) y / sHeight;	// invert?

	cx = cOffsetX + pxR * rWidth;
	cy = cOffsetY + pyR * hWidth;

	rWidth /= kZoomFactor;

	ResetRender();
	isZooming = true;
	TextureImage *tmp = fgWorld;
	fgWorld = bgWorld;
	bgWorld = tmp;
	startZoom = m_fTime;
	zoomDuration = 1;
	zoomOut = false;

	bgWorld->SetupZoom(0, sWidth/2 - x*kZoomFactor,
										 0, sHeight/2 - y*kZoomFactor,
										 sWidth, sWidth*kZoomFactor,
										 sHeight, sHeight*kZoomFactor,
										 1,1); // 1 1

	fgWorld->SetupZoom(x-sWidth/(kZoomFactor*2), 0,
										 y-sHeight/(kZoomFactor*2), 0,
										 sWidth/kZoomFactor, sWidth,
										 sHeight/kZoomFactor, sHeight,
										 0,1); // 0 1
}

void MandelWin::ZoomoutMandelbrot(int x, int y)
{
	int sWidth = glutGet(GLUT_WINDOW_WIDTH);
	int sHeight = glutGet(GLUT_WINDOW_HEIGHT);

	double	ocx = cx;
	double	ocy = cy;
	
	if (x > sWidth*3/8 && x < sWidth*5/8)	x = sWidth/2;
	if (y > sHeight*3/8 && y < sHeight*5/8) y = sHeight/2;

	double pxR = (double) x / sWidth;
	double pyR = (double) y / sHeight;	// invert?
	
	cx = cOffsetX + pxR * rWidth;
	cy = cOffsetY + pyR * hWidth;

	rWidth *= kZoomFactor;

	ResetRender();

	isZooming = true;

	TextureImage *tmp = fgWorld;
	fgWorld = bgWorld;
	bgWorld = tmp;
	startZoom = m_fTime;
	zoomDuration = 1;
	zoomOut = true;

  // these are currently backwards...
	// This is correct..
	bgWorld->SetupZoom(0, sWidth/2 - x/(kZoomFactor),
										 0, sHeight/2 - y/(kZoomFactor),
										 sWidth, sWidth/kZoomFactor,
										 sHeight, sHeight/kZoomFactor,
										 1,0);
	
	// Starting position has old cx,cy centered...
	double	ox = (ocx - cOffsetX)*sWidth/rWidth;
	double	oy = (ocy - cOffsetY)*sHeight/hWidth;
	
	fgWorld->SetupZoom(sWidth/2  - ox*kZoomFactor, 0, 
										 sHeight/2 - oy*kZoomFactor, 0,
										 sWidth*kZoomFactor,sWidth, 
										 sHeight*kZoomFactor, sHeight,
										 1,1);


	
}


bool MandelWin::HandleKey(int x)
{
	m_fLastKey = m_fTime;
	switch (tolower(x)) {
	case 's':
		showStats = !showStats;
		break;
	case 'c':
		cutOff *= 10;
		ResetRender();
		break;
	case 'x':
		cutOff /= 10;
		ResetRender();
		break;
	case 'p':
		TakeSnapshot();
		break;
	case 'f':
		ToggleFullscreen();
		break;
	case 'j':
		isJulia = !isJulia;
		if (isJulia) {
			jx = cx;
			jy = cy;
		}
		else {
			cx = jx;
			cy = jy;
		}
		ResetRender();
		break;
	case ' ':
			cAdjustComponent = (cAdjustComponent+1) % 3;
			break;
	case '-':	cPhase[cAdjustComponent] -= .1;	InitColors(); break;
	case '+': cPhase[cAdjustComponent] += .1;	InitColors();	break;
	case '[':	cFreq[cAdjustComponent] /= 1.5;	InitColors();	break;
	case ']':	cFreq[cAdjustComponent] *= 1.5;	InitColors();	break;
	}
	return true;
}

TextureImage::TextureImage(int w, int h, int glType)
{
	width = w;
	height = h;
	if (glType == GL_RGB)
		bpp = 24;
	else
		bpp = 32;

	rowBytes = w * (bpp/8);
	
	imageData = (unsigned char *) malloc(h * rowBytes);
	memset(imageData, 0, h*rowBytes);


	// Build A Texture From The Data
	glGenTextures(1, &texID);					// Generate OpenGL texture IDs

	glBindTexture(GL_TEXTURE_2D, texID);			// Bind Our Texture

	GLfloat bColor[] = {.5,.5,.5,0};

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);	// Linear Filtered
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);	// Linear Filtered
	glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR,bColor);	// Linear Filtered
	
	glTexImage2D(GL_TEXTURE_2D, 0, glType, width, height, 0, glType, GL_UNSIGNED_BYTE, imageData);
}

TextureImage::~TextureImage()
{
		glDeleteTextures(1, &texID);
		free(imageData);
}

void TextureImage::SetupZoom(GLfloat sx, GLfloat ex,
								 GLfloat sy, GLfloat ey,
								 GLfloat sw, GLfloat ew,
								 GLfloat sh, GLfloat eh,
								 GLfloat sa, GLfloat ea)
{
	scx = sx;	dcx = ex - sx;
	scy = sy;	dcy = ey - sy;
	scw = sw;	ecw = ew; 				dcw = ew - sw;	rcw = ew/sw;
	sch = sh;	ech = eh;					dch = eh - sh;	rch = eh/sh;
	this->sa = sa; this->da = ea - sa;
}



void MandelWin::ResetRender()
{
	int sWidth = glutGet(GLUT_WINDOW_WIDTH);
	int sHeight = glutGet(GLUT_WINDOW_HEIGHT);

	demmDelta = (rWidth*0.5)/(sWidth-1);
	pixelsPerPass = 12000;
	isDone = false;
	pX = 0;
	pY = 0;
	resolution = pRes = 16;
	resMask = 0;
	curLimit = recurseFactor*(-log10(rWidth));
	if (curLimit < MinIter)
		curLimit = MinIter;
	if (curLimit > MaxIter)
		curLimit = MaxIter;

	hWidth = rWidth*sHeight/(double)sWidth;
	cOffsetX = cx - rWidth/2;
	cOffsetY = cy - hWidth/2;
	hFactor = rWidth / kDefWidth;
	vFactor = hWidth / kDefWidth;
	startRender = m_fTime;
}

void MandelWin::GetCPixel(int x, int y, MyARGB *argb)
{
		unsigned long *dp = (unsigned long *) &fgWorld->imageData[y*fgWorld->rowBytes + (x<<2)];
		*((unsigned long *) argb) = *dp;
}


double	xOrbit[MaxIter];
double	yOrbit[MaxIter];

int	MandelWin::ComputePixelColor(double px, double py)
{
	double	x,y,x1,y1,p0,q0;
	int	kol,limit;

	// Note for julia, x,y = pixel coordinate, and p0,q0 = jx,jy

	p0 = hFactor*px + cOffsetX;
	q0 = vFactor*py + cOffsetY;
	x=p0;
	y=q0;	// Precompute 1 cycle... (normally x = y = 0)
	if (isJulia) {
		p0 = jx;
		q0 = jy;
	}
	x1 = x*x;
	y1 = y*y;

	limit = curLimit;
	kol = 0;
	xOrbit[0] = 0;
	yOrbit[0] = 0;

	do {
		x1 = x1 - y1 + p0;
		y1 = 2*x*y + q0;
		x = x1;
		y = y1;
		x1 *= x1;
		y1 *= y1;


		++kol;

		if (renderMode == RM_DEMM) {
			xOrbit[kol] = x;
			yOrbit[kol] = y;
		}

	} while (kol <= limit && x1+y1 < cutOff);

	// black interior
	if (kol > curLimit)
		return MaxColors;

	switch (renderMode) {
	// case RM_BlackFill:	return (kol & 1)? MaxColors+1 : MaxColors;
	case RM_SineCPM:
		{
			double pot = kol;
			double r = fabs(x1+y1);
			pot = (pot <= 0)? 0 : log(r)/pow(2,pot);
			pot = pot > 0? curLimit - pow(pot,1/(double) curLimit)*100 : MaxColors;
			if (pot > MaxColors)
				pot = MaxColors;
			return (int) pot;
		}
		break;
	case RM_SineToR:				return (kol*MaxColors)/curLimit;
	case RM_SineLSM:				return kol % MaxColors;
	case RM_Tiger:					return (kol & 1)? kol % MaxColors : (kol+4) % MaxColors;
	case RM_TigerCon:				return (kol & 1)? kol % MaxColors : (kol+32) % MaxColors;
	case RM_BinaryDecomp:		return y < 0? (kol+32) % MaxColors : kol % MaxColors;
/*	case RM_ATan:				
		{
			return MaxColors/2 + (int) (atan(x1/y1)*.25*MaxColors);
		}
		break;
	case RM_Test: // binary decomposition
		return x < y? kol % MaxColors : MaxColors - (kol % MaxColors);
 */
	case RM_DEMM:			// thick filament...
		{
			short	i;
			double	xder=0,yder=0,temp;
			bool	flag = 0;
			for (i = 0; i < kol && !flag; ++i) {
				temp = 2 * (xOrbit[i]*xder - yOrbit[i]*yder)+1;
				yder = 2 * (yOrbit[i]*xder + xOrbit[i]*yder);
				xder = temp;
				if (fabs(xder) > fabs(yder))
					flag = fabs(xder) > 1000000000;
				else
					flag = fabs(yder) > 1000000000;
			}
			if (!flag) {
				return (log(x1+y1)*sqrt(x1+y1)/sqrt(xder*xder+yder*yder)) < demmDelta? MaxColors : MaxColors+1;
			}
			else
				return MaxColors;
		}
	}
	return 0;
}

void MandelWin::IdleUpdate()
{
	if (mouseDownFlags & kLeftButton) {
		if (!isZooming)
			ZoominMandelbrot(mx, my);
	}
	else if (mouseDownFlags & kRightButton) {
		if (!isZooming)
			ZoomoutMandelbrot(mx, my);
	}
	
	bool				yOddBoundary = resMask == 0? 1 : (pY & resMask) != 0;
	int					pWidth = kDefWidth;
	int					pHeight = kDefWidth;
	int					minY = pY, maxY = pY+1;

/*	if (m_sleepAdjust > 10)
		++pixelsPerPass;
	else if (m_sleepAdjust == 0 && pixelsPerPass > 50)
		pixelsPerPass--;
*/

	int	n = pixelsPerPass / (isZooming? 2 : 1);
	MyRect	r;
	while (n-- && !isDone)
	{
		if (resolution == 1)
		{
				if (yOddBoundary || (pX & resMask) != 0) 
				{
					SetCPixel(pX, pY, &cTable[ComputePixelColor(pX,pY)]);
				}
		}
		else {
				if (yOddBoundary || (pX & resMask) != 0) 
				{
					MySetRect(&r,pX,pY,pX+pRes,pY+pRes);
					MyPaintRect(fgWorld, &r, &cTable[ComputePixelColor(pX,pY)]);
				}
		}

     // can do minRect calculations here to 
		// minimize texture gronking -- see MZoom
		pX += pRes;
		if (pX >= pWidth) 
		{
			pX = 0;
			pY += pRes;
			maxY = maxY < pY? pY : maxY;
			yOddBoundary = resMask == 0? 1 : (pY & resMask) != 0;
			if (pY >= pHeight) {
				pY = 0;
				pX = 0;
				maxY = pHeight;
				if (resolution == 1) 
				{
					isDone = true;
					resolution = 0;
					elapsedRender = m_fTime - startRender;
					break;
				}
				else {
					resMask = resolution - 1;
					resolution >>= 1;
					pRes = resolution? resolution : 1;
					yOddBoundary = 0;
				}
				if (minY > 0) // this is to minimize regions passed to glTexSubImage2D at lower resolutions.
					break;
				minY = 0;
			}
		}
	}
	
	glBindTexture(GL_TEXTURE_2D, fgWorld->texID);			// Bind Our Texture
	
	glTexSubImage2D(GL_TEXTURE_2D, 0,      0, minY, 
		              fgWorld->width, (maxY - minY), 
									GL_RGBA, 
		              GL_UNSIGNED_BYTE, &fgWorld->imageData[minY*fgWorld->rowBytes]);

	
//	glTexImage2D(GL_TEXTURE_2D, 0, glType, fgWorld->width, fgWorld->height, 0, glType, 
//		             GL_UNSIGNED_BYTE, fgWorld->imageData);
}

void MandelWin::ErrorMessage(char *str)
{
	strcpy(errMsgBuf, str);
}

void ErrorMessage(char *str,...)
{
	char tbuf[128];
	va_list args;
	va_start(args,str);
	vsprintf(tbuf,str,args);
	va_end(args);

	gMandelWin->ErrorMessage(tbuf);
}

void ErrorExit(char *str,...)
{
	char tbuf[128];
	va_list args;
	va_start(args,str);
	vsprintf(tbuf,str,args);
	va_end(args);

	gMandelWin->ErrorMessage(tbuf);
	exit(0);
}

#define pi	3.14159

MyARGB	exColors[ExtraColors] = {
	{	0x00,	0x00,	0x00	},
	{	0xDD,	0xDD,	0xCC	},
	{	0xFF,	0x00,	0x00	},
	{	0xFF,	0xFF,	0x00	},
	{	0x00,	0xFF,	0x00	},
	{	0x00,	0xFF,	0xFF	},
	{	0x00,	0x00,	0xFF	},
	{	0xFF,	0x00,	0xFF	}};


void MandelWin::InitColors(void)
{
	long			i;
	double		n;
	long			ln;
	MyHSV			hsv;
	double		hueM,valM,satM,redM,grnM,bluM;
	switch (paletteType) {
	case RM_HSVRamp:
		satM = (cFreq[0]*pi)/MaxColors;
		valM = (cFreq[1]*pi)/MaxColors;
		hueM = (cFreq[2]*pi)/MaxColors;
		for (i = 0; i < MaxColors; ++i) {
			n = (-cos(i*hueM + cPhase[0]) + 1) * 128;
			hsv.hue = (int) MyMax(0, MyMin(255, n));
			n = (cos(i*satM + cPhase[1]) + 1) * 128;
			hsv.saturation =  (int) MyMax(0, MyMin(255, n));
			n = (cos(i*valM + cPhase[2]) + 1) * 128;
			hsv.value =  (int) MyMax(0, MyMin(255, n));;
			HSVtoRGB(&hsv,&cTable[i]);
		}	
		break;
	case RM_RGBRamp:
		redM = (cFreq[0]*pi)/MaxColors;
		grnM = (cFreq[1]*pi)/MaxColors;
		bluM = (cFreq[2]*pi)/MaxColors;
		for (i = 0; i < MaxColors; ++i) {
			n = (cos(i*redM + cPhase[0]) + 1) * 128;
			cTable[i].r = (int) MyMax(0, MyMin(255, n));
			n = (-cos(i*grnM+ cPhase[1]) + 1) * 128;
			cTable[i].g = (int) MyMax(0, MyMin(255, n));
			n = (-cos(i*bluM+ cPhase[2]) + 1) * 128;
			cTable[i].b = (int) MyMax(0, MyMin(255, n));
		}	
		break;
	case RM_Grayscale:
		{
			double	vm = (cFreq[0]*pi)/MaxColors;
			for (i = 0; i < MaxColors; ++i) {
				n = (-cos(i*vm + cPhase[0])+1)*128;
				ln =  (int) MyMax(0, MyMin(255, n));
				cTable[i].r = ln;
				cTable[i].g = ln;
				cTable[i].b = ln;
			}	
		}
		break;
	}
	memcpy(&cTable[MaxColors],&exColors[0],sizeof(MyARGB)*ExtraColors);
	ResetRender();	
}